# Hardware Debugging in Visual Studio Code and WSL

Real debugging makes life easier, here is a configuration for VSCode with WSL for Windows 10/11.

## Hardware:
For the basics, please read [Hardware Debugging.md](Hardware%20Debugging.md) up to the section "Compilation options", everything after that is outdated.
In contrast to the instructions above, I had to connect the Target VCC pin of my original ST-Link/V2 to the 3.3V of the FC, otherwise I got an error message that the voltage of the target was too low. The FC is not supplied with voltage via this pin, only the voltage of the target is measured to ensure signal compatibility. (See ST-Link/V2 manual, table 4).
Check with "STM32 ST-LINK Utility" if the connection to the FC is working.

## Software:
Read [IDE - Visual Studio Code with Windows 10.md](IDE%20-%20Visual%20Studio%20Code%20with%20Windows%2010.md) and set up VSCode like this. 
I recommend to use WSL2 and to work in the Linux file system (e.g. `~/git/inav`), access to the files in the Linux FS you get
in Windows 10 via `\\wsl$` in the address line of the explorer, in Windows 11 via the new entry "Linux" in the explorer.

"Cortex Debug" is also required as an additional extension for VSCode.

## OpenOCD
Now we come to the sticking point:
Unfortunately, WSL cannot pass the ST link (or only via tools such as usbipd-win, see the instructions here: [Windows 11 - VS Code - WSL2 - Hardware Debugging.md](Windows%2011%20-%20VS%20Code%20-%20WSL2%20-%20Hardware%20Debugging.md)), so OpenOCD must run on the Windows side, which is no problem since gdb and OpenOCD communicate with each other via TCP. 

Tip:
Only for OpenOCD you don't need to install xpm and node.js, just download the archive at https://github.com/xpack-dev-tools/openocd-xpack/releases and unpack it into a folder of your choice.

Now OpenOCD needs board definition files for the FC, the files generated by the makefile do not work for me. 
Copy the following files to `scripts\board` in the OpenOCD folder:

f4fc.cfg:
```
# Boardconfig for ST-Link/V2 with F4-FC

source [find interface/stlink.cfg]

transport select hla_swd

source [find target/stm32f4x.cfg]

reset_config none separate
```

f7fc.cfg
```
# Boardconfig for ST-Link/V2 with F7-FC

source [find interface/stlink.cfg]

transport select hla_swd

source [find target/stm32f7x.cfg]

reset_config none separate
```

h7fc.cfg
```
# Boardconfig for ST-Link/V2 with H7-FC

source [find interface/stlink.cfg]

transport select hla_swd

source [find target/stm32h7x_dual_bank.cfg]

reset_config none separate
```

In order for GDB and OpenOCD to be able to communicate with each other, the host (Windows) IP address is required, the easiest and most convenient way is to have this done automatically by the linux bash to do this automatically, add the following line to the `~/.bashrc` file:

`export WSL_HOST_IP=$(cat /etc/resolv.conf | sed -rn 's|nameserver (.*)|\1|p')`

Actually, the trick is quite simple:
In launch.json set `servertype` to `external` and `gdbTarget` to `windows-host-ip:3333`, the .cfg file must of course be passed when starting openOCD in windows and the connection must be bound to 0.0.0.0.

## VSCode configuration files

For convenience, I have created a set of VSCode configuration files, that do all the magic.
And before anyone asks: Yes, WSL can launch applications on the host.

launch.json
```
{
    "configurations": [
      {
        "name": "Debug",
        "type": "cortex-debug",
        "request": "launch",
        "servertype": "external",
        "cwd": "${workspaceRoot}/debug",
        //"runToEntryPoint": "main",
        "executable": "${workspaceRoot}/debug/bin/${config:inav.debug.target}.elf",
        "gdbTarget": "${config:openocd.host}:3333",
        "svdFile": "${workspaceRoot}/dev/svd/${config:inav.debug.svdFile}",
        "preLaunchTask": "Debug",
        "showDevDebugOutput": "parsed"
      },
    ],
  }
```

tasks.json
```
{
    "version": "2.0.0",
    "tasks": [
        {
            "label": "Launch OpenOCD",
            "command": "${config:openocd.path}",
            "args": [
                "-f",
                "\"board/${config:inav.debug.board}\"",
                "-c",
                "\"bindto 0.0.0.0\""
            ],
            "type": "shell",
            "isBackground": false,
            "group": "none",
            "presentation": {
                "reveal": "always",
                "panel": "new"
            },
            "problemMatcher": []
        },
        {
            "label": "CMAKE Release",
            "type": "shell",
            "command": "mkdir -p release && cd release && cmake -DCMAKE_BUILD_TYPE=Release ..",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}"
            }
        },
        {
            "label": "CMAKE Debug",
            "type": "shell",
            "command": "mkdir -p debug && cd debug && cmake -DCMAKE_BUILD_TYPE=Debug ..",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}"
            }
        },
        {
            "label": "CMAKE Build",
            "type": "shell",
            "command": "mkdir -p build && cd build && cmake ..",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}"
            }
        },
        {
            "label": "Compile autogenerated docs",
            "type": "shell",
            "command": "python3 src/utils/update_cli_docs.py",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}"
            }
        },
        {
            "label": "Debug",
            "type": "shell",
            "command": "make ${config:inav.debug.target}",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}/debug"
            }
        },
        {
            "label": "Release",
            "type": "shell",
            "command": "make ${config:inav.release.target}",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}/release"
            }
        },
        {
            "label": "Build",
            "type": "shell",
            "command": "make ${config:inav.release.Target}",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}/build"
            }
        },
        {
            "label": "Clean Debug",
            "type": "shell",
            "command": "make clean",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}/debug"
            }
        },
        {
            "label": "Clean Build",
            "type": "shell",
            "command": "make clean",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}/build"
            }
        },
        {
            "label": "Clean Release",
            "type": "shell",
            "command": "make clean",
            "group": "build",
            "problemMatcher": [],
            "options": {
                "cwd": "${workspaceFolder}/release"
            }
        },
    ]
}
```

settings.json
```
{
    "cortex-debug.armToolchainPath": "${workspaceRoot}/tools/gcc-arm-none-eabi-10-2020-q4-major/bin",
    "openocd.host": "${env:WSL_HOST_IP}",
    "openocd.path": "/mnt/path/to/openocd/bin/openocd.exe",
    "inav.debug.svdFile": "STM32F405.svd",
    "inav.debug.target": "OMNIBUSF4SD",
    "inav.release.target": "MATEKF722SE",
    "inav.debug.board": "f4fc.cfg"
}
```

Only the settings.json needs to be modified:

| Option | Explanation |
|--------|-------------|
| cortex-debug.armToolchainPath | path to ARM toolchain, e.g. `${workspaceRoot}/tools/gcc-arm-none-eabi-10-2020-q4-major/bin` |
| openocd.host | IP address of the Windows host, if the environment variable was set as above, do not change anything here, otherwise manually enter the IP of the Windows host. |
| openocd.path | Path to OpenOCD in Windows, format `/mnt/[windows-drive-letter]/path/to/openocd.exe`, for example `/mnt/c/openocd-0.11.0-1/bin/openocd.exe`. |
| inav.debug.svdFile | SVD file, accroding to the processor on your board, see `/dev/svd`, for example `STM32F405.svd`. |
| inav.debug.target | Debug target, for example `OMNIBUSF4SD` |
| inav.release.target | Release target, for example `MATEKF722SE` |
| inav.debug.board | cfg file for OpenOCD, matching the processor of the board, e.g. `f4fc.cfg`, see above |

### Tasks

The following tasks are available
| Task | Description |
|------|-------------|
| Launch OpenOCD | Must be started manually once per session, starts OpenOCD with the correct options. |
| CMAKE Release | Creates a new folder "release", and configures the buildsystem for a real release build in it. |
| CMAKE Debug | Creates a new folder "debug", and configures the buildsystem for a real debug build in it. |
| CMAKE Build | Creates a new folder "build" and configures the buildsystem for the standard build. |
| Compile autogenerated docs | Rebuilds the CLI documentation |
| Debug | Compiles a new debug build for the target specified in inav.debug.target |
| Release | Compiles a new release build for the target specified in inav.release.target. |
| Build | Compiles a new standard build for the target specified in inav.release.target. |
| Clean Debug | Cleans up the debug folder |
| Clean Release | Cleans up the release folder |
| Clean Build | Cleans up the build folder |
 
With `F5`, a new debug session can be started normally (recompiling if necessary), with `CTRL+SHIFT+B` a build task can be started.

## Problem solving
If OpenOCD reports an error `Couldn't bind to ...`, it may be that Hyper-V is blocking these ports, see this thread on stackoverflow for solutions:
https://stackoverflow.com/questions/48478869/cannot-bind-to-some-ports-due-to-permission-denied
